/*
$Id: ir.cpv 1.2 1994/10/31 21:45:38 chris Exp chris $

Program to record IR remote control signals

Chris Dodge - April 94
*/

#include <dos.h>
#include <conio.h>
#include <graphics.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys\timeb.h>
#include "..\lib\irutils.h"
#include "ir.h"

// -- Global vars --
//char Signal[SIZE];
int  GraphDone = FALSE;
int  ClkCount;
struct DeviceData DeviceDat;   // Holds data above device signals
struct SigDat SD[SIGNO];       // Can hold upto SIGNO signals for a device
char PTITLE[40];

// -- ASM functions --
extern int RecIR(char*, int);    // ASM - Read IR signal
extern int PlayIR(char*, int);   // ASM - Play signal back

#define ONE "1000"
#define ZERO "10"

int main(void)
{
	CreateTitleString();
	MainMenu();        // Call main menu....
	clrscr();
	exit(0);
}

void CreateTitleString()
{
	char revstr[] = "$Revision: 1.2 $";
	char tempstr[12];
	int minr, majr;
	sscanf(revstr, "%s %d.%d", tempstr, &majr, &minr);
	sprintf(PTITLE, " PC IR Remote Control - V %d.%d ", majr, minr);
}

// Records signal
void DoRecord() {
	int SigNo;            // Signal number
	int HL;               // Highlighted data char no.
	int Off;              // Offset of calculated signal (for alignment)
	int i, j, err;
	char c;

	SigNo = 0; HL = 0; Off = 0;
	err=InitGraph();
	cleardevice();
	ClearSignalArray();

	// Draw results etc.
	cleardevice();
	DrawRes(WHITE, 0, 0x0001);     // Draw captured signal
	PrintParams(SigNo, LIGHTRED, HL);
	PrintInst(LIGHTBLUE);
	CalcSignal(0x0002, SigNo, Off);
	DrawRes(LIGHTRED, 10, 0x0002);
	c = getch();

	while (c != 'q') {
		switch (c) {

			// Increase the clock count
			case 'f': case 'F':
				DrawRes(BLACK, 10, 0x0002);
				PrintParams(SigNo, BLACK, HL);
				if (c == 'F') DeviceDat.Clock+=10;
				else DeviceDat.Clock++;
				PrintParams(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Decrease the clock count
			case 'g': case 'G':
				DrawRes(BLACK, 10, 0x0002);
				PrintParams(SigNo, BLACK, HL);
				if (c == 'G') DeviceDat.Clock -= 10;
				else DeviceDat.Clock--;
				PrintParams(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Increase the bias value
			case 'b': case 'B':
				DrawRes(BLACK, 10, 0x0002);
				PrintParams(SigNo, BLACK, HL);
				if (c == 'B') DeviceDat.Bias+=10;
				else DeviceDat.Bias++;
				PrintParams(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Decrease the bias value
			case 'c': case 'C':
				DrawRes(BLACK, 10, 0x0002);
				PrintParams(SigNo, BLACK, HL);
				if (c == 'C') DeviceDat.Bias-=10;
				else DeviceDat.Bias--;
				PrintParams(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Move highlighted point 1 <--
			case 75:
				if (HL == 0) break;
				HL--;
				PrintDataStr(SigNo, LIGHTRED, HL);
				break;

			// Move highlighted point 1 -->
			case 77:
				if (HL >= DeviceDat.SignalLen) break;
				HL++;
				PrintDataStr(SigNo, LIGHTRED, HL);
				break;

			// Toggle data value
			case 13:
				PrintDataStr(SigNo, BLACK, HL);
				DrawRes(BLACK, 10, 0x0002);
				if (SD[SigNo].data[HL] == '1') SD[SigNo].data[HL] = '0';
				else SD[SigNo].data[HL] = '1';
				PrintDataStr(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Increase start pos
			case 's': case 'S':
				DrawRes(BLACK, 10, 0x0002);
				Off+=2;
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Decrease start pos
			case 't': case 'T':
				DrawRes(BLACK, 10, 0x0002);
				Off-=2;
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Next signal
			case 'n': case 'N':
				if (SigNo == (DeviceDat.SignalNo-1)) break;
				PrintParams(SigNo, BLACK, HL);
				DrawRes(BLACK, 10, 0x0002);
				SigNo++;
				PrintParams(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Previous signal
			case 'p': case 'P':
				if (SigNo == 0) break;
				PrintParams(SigNo, BLACK, HL);
				DrawRes(BLACK, 10, 0x0002);
				SigNo--;
				PrintParams(SigNo, LIGHTRED, HL);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Play the signal
			case 'r': case 'R':
				i = PlayIR(Signal, SIZE);
				break;

			// Write the data to file
			case 'w': case 'W':
				i = SaveData(DeviceDat.Name);
				break;

			// Take new signal input
			case 'i': case 'I':
				cleardevice();
				printf("*** INPUT SIGNAL %s NOW ***\n", SD[SigNo].Name);
				i = RecIR(Signal, SIZE);   // Call ASM routine
				cleardevice();
				DrawRes(WHITE, 0, 0x0001);     // Draw captured signal
				PrintParams(SigNo, LIGHTRED, HL);
				PrintInst(LIGHTBLUE);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				break;

			// Try to interpret the signal (e for estimate)
			case 'e': case 'E':
				DrawRes(BLACK, 10, 0x0002);
				PrintDataStr(SigNo, BLACK, HL);
				CalcData(0x0001, SigNo, Off);
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				PrintDataStr(SigNo, LIGHTRED, HL);
				break;

			// Increase pause between signal repeats
			case 'y': case 'Y':
				DrawRes(BLACK, 10, 0x0002);
				PrintParams(SigNo, BLACK, HL);
				if (c == 'Y') DeviceDat.Pause+=10;
				else DeviceDat.Pause++;
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				PrintParams(SigNo, LIGHTRED, HL);
				break;

			// Decrease pause between signal repeats
			case 'x': case 'X':
				DrawRes(BLACK, 10, 0x0002);
				PrintParams(SigNo, BLACK, HL);
				if (c == 'X') DeviceDat.Pause-=10;
				else DeviceDat.Pause--;
				CalcSignal(0x0002, SigNo, Off);
				DrawRes(LIGHTRED, 10, 0x0002);
				PrintParams(SigNo, LIGHTRED, HL);
				break;

			default:
				break;
		}
		c = getch();

	}

	return;
}

// Draw the signal on the screen
// Col = Colour, Pos = vertical offset for drawing
// Mask = Bit in char to be drawn
void DrawRes(int Col, int Pos, int Mask)
{
	int i, j, x, y, NoRows, VertOff;
	int Lasty, Len;
	float f;

	setcolor(Col);
	NoRows = 11;
	Len = 1280*2;
	for (i=0; i<NoRows; i++) {
		VertOff=((i+1)*25)+Pos;
		for (j=0; j<Len; j++) {
			if ((Signal[(i*Len)+j]&Mask) != 0) y = VertOff;
			else y = VertOff-6;
			x = j/4;
			if (Lasty!=y) line(x,y,x,Lasty);
			putpixel(x, y, Col);
			Lasty = y;
		}
	}

	return;
}

int InitGraph()
{
	int err=0;
	int graphdriver = DETECT, graphmode;

	clrscr();

	// Register graphics stuff, first time round only.
	if (!GraphDone) {
		err = registerbgidriver(EGAVGA_driver);
		if (err < 0) GraphErr(err);

		initgraph(&graphdriver, &graphmode, NULL);
		err = graphresult();
		if (err != grOk) GraphErr(err);
	}
	else setgraphmode(getgraphmode());

	GraphDone = TRUE;

	return err;
}

// Report graphics error and exit.
void GraphErr(int err)
{
	fprintf(stderr, "*** Graphics Error %s ***\n", grapherrormsg(err));
	printf("Hit a key man....\n");
	getch();
	exit(1);
}

// Read the data from file for this particular device
int ReadData(char* Dev)
{
	FILE     *data;          // Pointer to data file
	char    line[250];       // Input line
	char    dat[250];        // Temp var for data reading
	char    ThisDev[8];      // The device type of current line
	char    Info[16];        // Information type in current line
	char    SigName[20];     // Temp var for reading signal name
	char    ch;              // Temp var ..
	int     ChPos;           // Input character position counter
	char    type;            // The type of current word.
	int     NoFields;        // Number of fields read by scanf.
	int     SigCount;        // Number of signals read
	char    N;               // Current signal name
	char    S;               // Current signal
	int     ValidityCount;   // Checks have enogh parameters
	int     I, i;
	float   F;

	if ((data = fopen("irinfo.dat", "r")) == NULL) {
		fprintf(stderr, "*** Cannot open file IRINFO.DAT ***\n");
		return 1;
	}

	PrintMsg("Loading signal data...\0");
	sprintf(DeviceDat.Name, Dev);
	SigCount = 0; ValidityCount = 0;

	// Read the data file.....
	do {

		// Read each line....
		ch = fgetc(data);
		ChPos = 0;
		type = 0;
		while ((ch!='\n')&&(ChPos<(SIGLEN+18))&&(ch!=EOF)) {
			line[ChPos] = ch;
			ChPos++;
			ch = fgetc(data);
		}
		line[ChPos] = '\0';       // Null-terminate line.

		// Cancel previous values otherwise they stay on NULL lines
		// ie. they are not cleared by scanf
		sprintf(ThisDev, "\0");
		sprintf(Signal, "\0");

		// Extract the device type and info type
		NoFields = sscanf(line, "%s %s", ThisDev, Info);

		// If this device is of the wanted type, then read data
		// else ignore.
		if (!(strncmpi(ThisDev, Dev, strlen(Dev)))) {

			// Act on the word type option.
			if (!(strcmpi(Info, "Clock"))) {
				NoFields = sscanf(line, "%s %s %f", ThisDev, Info,
													&DeviceDat.Clock);
				ValidityCount++;
			}

			if (!(strcmpi(Info, "Bias"))) {
				NoFields = sscanf(line, "%s %s %d", ThisDev, Info,
													&DeviceDat.Bias);
				ValidityCount++;
			}

			if (!(strcmpi(Info, "SignalNo"))) {
				NoFields = sscanf(line, "%s %s %d", ThisDev, Info,
													&DeviceDat.SignalNo);
				ValidityCount++;
			}

			if (!(strcmpi(Info, "SignalLen"))) {
				NoFields = sscanf(line, "%s %s %d", ThisDev, Info,
													&DeviceDat.SignalLen);
				// Check length value
				if (DeviceDat.SignalLen > SIGLEN) {
					PrintMsg("*** ERROR: Required signal length too long ***");
					getch();
					exit(1);
				}
				ValidityCount++;
			}

			if (!(strcmpi(Info, "Pause"))) {
				NoFields = sscanf(line, "%s %s %d", ThisDev, Info,
													&DeviceDat.Pause);
				ValidityCount++;
			}

			if (!(strcmpi(Info, "Repeat"))) {
				NoFields = sscanf(line, "%s %s %d", ThisDev, Info,
													&DeviceDat.Repeat);
				ValidityCount++;
			}

			if (!(strcmpi(Info, "Signal"))) {
				if (SigCount == SIGNO) {
					PrintMsg("*** Error: Too many signals ***");
					printf("%s\n", line);
					getch();
				}

				else {
					NoFields = sscanf(line, "%s %s %s %s", ThisDev, Info,
														SigName, dat);

					// Check length of signal name
					if ((strlen(SigName)>=SIGNAMELEN)) SigName[SIGNAMELEN-1] = '\0';
					sprintf(SD[SigCount].Name, "%s", SigName);
					if (NoFields < 4) strcpy(dat, "");

					// Check signal data length, if too short, fill with 0's
					if ((strlen(dat)<DeviceDat.SignalLen)) {
						for (i=(strlen(dat)-1); i<(DeviceDat.SignalLen-1); i++)
							dat[i] = '0';
						dat[DeviceDat.SignalLen] = '\0';
					}

					// If too long, terminate
					if ((strlen(dat)>DeviceDat.SignalLen)) {
						dat[DeviceDat.SignalLen] = '\0';
					}

					// Copy dat --> signal data struct
					sprintf(SD[SigCount].data, "%s", dat);

					SigCount++;
				}
			}
		}
	} while (ch != EOF);

	fclose(data);

	if (SigCount == 0) {
		PrintMsg("No signals for device - select another!");
		return 1;
	}

	sprintf(line, "%d signal(s) loaded.\0", SigCount);
	PrintMsg(line);

	if (ValidityCount!=6) {
		PrintMsg("Not enough parameter data for device - select another!");
		return 1;
	}

	return 0;

}


// Save the data to file for this particular device
int SaveData(char* Dev)
{
	FILE     *data, *out;    // Pointer to data and output files
	char    line[SIGLEN+20]; // Input line
	char    ThisDev[8];      // The device type of current line
	char    Info[16];        // Information type in current line
	char    ch;              // Temp var ..
	int     ChPos;           // Input character position counter
	char    type;            // The type of current word.
	int     NoFields;        // Number of fields read by scanf.
	int     SigCount;        // Number of signals read
	char    N;               // Current signal name
	char    S;               // Current signal
	int     I, i;
	float   F;

	// Open info file
	if ((data = fopen("irinfo.dat", "r")) == NULL) {
		fprintf(stderr, "*** Cannot open file IRINFO.DAT ***\n");
		return 1;
	}
	// Open temp write file
	if ((out = fopen("irtemp.dat", "w")) == NULL) {
		fprintf(stderr, "*** Cannot open file IRTEMP.DAT ***\n");
		return 1;
	}

	SigCount = 0;

	// Read the data file.....
	do {

		// Read each line....
		ch = fgetc(data);
		ChPos = 0;
		type = 0;
		while ((ch!='\n')&&(ChPos<(SIGLEN+20))&&(ch!=EOF)) {
			line[ChPos] = ch;
			ChPos++;
			ch = fgetc(data);
		}
		line[ChPos] = '\0';       // Null-terminate line.

		// Extract the device type and info type
		if (strlen(line)>6)
			NoFields = sscanf(line, "%s %s", ThisDev, Info);
		else
			sprintf(ThisDev, "  ");

		// If this device is of the wanted type, then read data
		// else ignore.
		if (!(strncmpi(ThisDev, Dev, strlen(Dev)))) {

			// Act on the word type option.
			if (!(strcmpi(Info, "Clock"))) {
				sprintf(line, "%s Clock %f", DeviceDat.Name, DeviceDat.Clock);
			}

			if (!(strcmpi(Info, "Bias"))) {
				sprintf(line, "%s Bias %d", DeviceDat.Name, DeviceDat.Bias);
			}

			if (!(strcmpi(Info, "SignalNo"))) {
				sprintf(line, "%s SignalNo %d", DeviceDat.Name, DeviceDat.SignalNo);
			}

			if (!(strcmpi(Info, "SignalLen"))) {
				sprintf(line, "%s SignalLen %d", DeviceDat.Name, DeviceDat.SignalLen);
			}

			if (!(strcmpi(Info, "Pause"))) {
				sprintf(line, "%s Pause %d", DeviceDat.Name, DeviceDat.Pause);
			}

			if (!(strcmpi(Info, "Repeat"))) {
				sprintf(line, "%s Repeat %d", DeviceDat.Name, DeviceDat.Repeat);
			}

			if (!(strcmpi(Info, "Signal"))) {
				SD[SigCount].data[DeviceDat.SignalLen] = '\0';
				sprintf(line, "%s Signal %s %s", DeviceDat.Name,
												SD[SigCount].Name, SD[SigCount].data);
				SigCount++;
			}
		}

		// Write this line to op
		for (i=0; i<strlen(line); i++)
			putc(line[i], out);
		putc('\n', out);

	} while (ch != EOF);

	fclose(data);
	fclose(out);

	return 0;

}


// Prints various params associated with the IR signal for a device
// sig = signal no. HL = highlighted char in data printout
void PrintParams(int sig, int Col, int HL)
{
	char op[100];

	setcolor(Col);
	sprintf(op, "DEVICE       - %s", DeviceDat.Name);
	outtextxy(10, 360, op);
	sprintf(op, "Clock freq   - %.1f Hz", DeviceDat.Clock);
	outtextxy(10, 370, op);
	sprintf(op, "Hi/Low bias  - %d", DeviceDat.Bias);
	outtextxy(10, 380, op);
	sprintf(op, "Repeat No.   - %d", DeviceDat.Repeat);
	outtextxy(10, 390, op);
	sprintf(op, "Repeat Pause - %d", DeviceDat.Pause);
	outtextxy(10, 400, op);
	sprintf(op, "Signal name  - %s", SD[sig].Name);
	outtextxy(10, 410, op);

	// Print out data string
	sprintf(op, "Data        - ");
	outtextxy(10, 420, op);
	PrintDataStr(sig, Col, HL);

	return;
}

// Print out the data string. HL = highlighted char
void PrintDataStr(int Sig, int Col, int HL)
{
	int NoRows, i, j, x, y, Off, This;
	char str[2];

	str[1] = '\0';
	setcolor(Col);
	NoRows = (DeviceDat.SignalLen/60)+1;
	for (i=0; i<NoRows; i++)
		for (j=0,y=420+(i*10),x=120,Off=(i*60); j<60; j++,x+=8) {
			This = Off+j;
			if (This<DeviceDat.SignalLen) {
				str[0] = SD[Sig].data[This];

				// Check for highlighted char
				if (This == HL) {
					if (Col == BLACK) setcolor(Col);
					else setcolor(LIGHTBLUE);
					outtextxy(x,y,str);
					setcolor(Col);
				}
				else outtextxy(x,y,str);

			} else return;
		}

	return;
}

// Calculate signal from file data. No = Signal no.
// SO = Starting offset (for alignment)
void CalcSignal(int Mask, int No, int SO)
{
	float NoCounts;      // No. of counts in Sig array per IR up or down
	int Offset;          // No. of counts so far thru array
	int i, j, r, Val, Clear, From, To, SBias, EBias, PrevData;
	int Pos;

	NoCounts = SAMPLEFREQ/DeviceDat.Clock;
	Clear = 0xFF - Mask;
	PrevData = 0;

	// Loop over no. of signal repeats
	for (r=0; r<DeviceDat.Repeat; r++) {

		// Loop over signal...
		for (i=0,Offset=0; i<SIGLEN; i++) {

			Pos = i+(r*(SIGLEN+DeviceDat.Pause));

			// Determine whether a 1 is placed in array or not
			if (SD[No].data[i] == '1') {
				Val = 0;
			} else {
				Val = Mask;
			}

			// Work out biasing (ie. signal high/low irregularity)
			SBias = 0; EBias = 0;
			if ((SD[No].data[i]=='1')) EBias = DeviceDat.Bias;
			if ((PrevData=='1') && (SD[No].data[i]=='0')) {
				SBias = DeviceDat.Bias;
			}

			From = (int)(NoCounts*Pos) + SO + SBias;
			To = (int)(NoCounts*(Pos+1)) + SO + EBias;
			if (From < 0) From = 0;
			if (To > SIZE) To = SIZE;
			if (From >= SIZE) return;

			for (j=From; j<To; j++) {
				if (j<SIZE) {         // Check bounds
					Signal[j] &= Clear;
					Signal[j] |= Val;
				}
			}

			PrevData = SD[No].data[i];
		}

		// Fill up gap between this signal and the repeat with 0's
		for (i=j; i<(j+DeviceDat.Pause*NoCounts); i++)
			if (i<SIZE) {         // Check bounds
				Signal[i] &= Clear;
				Signal[i] |= Val;
			}
	}

	return;
}

// Calculate data from signal. No = Signal no.
// SO = Starting offset (for alignment)
void CalcData(int Mask, int No, int SO)
{
	float NoCounts;      // No. of counts in Sig array per IR up or down
	int Offset;          // No. of counts so far thru array
	int i, j, From, To, SBias, EBias, PrevData;
	int One, Zero;

	NoCounts = SAMPLEFREQ/DeviceDat.Clock;
	PrevData = 0;

	for (i=0,Offset=0; i<SIGLEN; i++) {

		// Work out biasing (ie. signal high/low irregularity)
		SBias = 0; EBias = 0;
		if ((SD[No].data[i]=='1')) EBias = DeviceDat.Bias;
		if ((PrevData=='1') && (SD[No].data[i]=='0')) {
			SBias = DeviceDat.Bias;
		}

		From = (int)(NoCounts*i) + SO + SBias;
		To = (int)(NoCounts*(i+1)) + SO + EBias;
		if (From < 0) From = 0;
		if (To > SIZE) To = SIZE;
		if (From >= SIZE) return;

		// For this part of the signal represented by data[i]
		// look to see if more 1's or 0's.
		for (j=From, One=0, Zero=0; j<To; j++) {
			if ((Signal[j] & Mask) == 0) One++;
			else Zero++;
		}
		PrevData = SD[No].data[i];

		// Adjust the values of Zero and One according to the
		// bias value
		Zero *= (float)((NoCounts+DeviceDat.Bias)/NoCounts);

		// Set the data value
		if (Zero > One) SD[No].data[i] = '0';
		else SD[No].data[i] = '1';

	}

	return;
}

// Print instructions for the data capture screen
void PrintInst(int Col)
{
	char op[30];

	setcolor(Col);
	sprintf(op, "Inc/Dec  Freq.   (f/g)");
	outtextxy(400, 320, op);
	sprintf(op, "Move LEFT/RIGHT  (>/<)");
	outtextxy(400, 330, op);
	sprintf(op, "Inc/Dec Start    (s/t)");
	outtextxy(400, 340, op);
	sprintf(op, "Inc/Dec Bias     (b/c)");
	outtextxy(400, 350, op);
	sprintf(op, "Next/Prev Signal (n/p)");
	outtextxy(400, 360, op);
	sprintf(op, "Write data to file (w)");
	outtextxy(400, 370, op);
	sprintf(op, "Input new signal   (i)");
	outtextxy(400, 380, op);
	sprintf(op, "QUIT               (q)");
	outtextxy(400, 400, op);

	sprintf(op, "Run the signal   (r/R)");
	outtextxy(200, 320, op);
	sprintf(op, "Toggle data val. (RET)");
	outtextxy(200, 330, op);
	sprintf(op, "Estimate signal. (e/E)");
	outtextxy(200, 340, op);
	sprintf(op, "Inc/Dec pause.   (y/x)");
	outtextxy(200, 350, op);
	return;

}

// Clear signal array
void ClearSignalArray()
{
	int i;

	for (i=0; i<SIZE; i++) Signal[i] = 0;
}

// Function to list possible signals for a device
// Rep = repeat signal playback or not?
void DoPlayback(int Rep)
{
	int i, j, no;

	PrintMsg("Hit any key to continue");
	getch();
	textbackground(BLACK);
	textcolor(LIGHTGRAY);
	window(1,1,80,25);
	clrscr();

	ClearSignalArray();

	// Print out signal list
	for (i=0; i<DeviceDat.SignalNo; i++) {
		if (i>=15) gotoxy(35,i-13);
		else gotoxy(5,i+2);
		cprintf("%d) %s", i, SD[i].Name);
	}

	gotoxy(5, 22);
	cprintf("Input a number (99 to quit) : ");
	_setcursortype(_NORMALCURSOR);
	no = 0;

	while (no != 99) {
		gotoxy(35, 22);
		scanf("%d", &no);
		fflush(stdin);
		gotoxy(35, 22);
		cprintf("      ");

		if ((no >= 0) && (no < DeviceDat.SignalNo)) {
			if (Rep) {
				textcolor(RED); gotoxy(25,24);
				cprintf("Hit any key to stop.....");
				_setcursortype(_NOCURSOR);
				CalcSignal(0x0002, no, 0);
				while (!(kbhit())) {
					textcolor(WHITE);
					gotoxy(55, 22); cprintf("SIGNAL");
					j = PlayIR(Signal, SIZE);
					textcolor(BLACK);
					gotoxy(55, 22); cprintf("SIGNAL");
					delay(1000);
				}
			} else {
				CalcSignal(0x0002, no, 0);
				j = PlayIR(Signal, SIZE);
			}
		}
		textcolor(BLACK); gotoxy(25,24);
		cprintf("Hit any key to stop.....");
		_setcursortype(_NORMALCURSOR);
	}

	return;
}

// Proc to find the signals for Tape player
int FindTapeIR()
{
	int StartVal, StopVal, SigNo;
	int i, j, k, err;
	long int StartGroup, StopGroup, c;
	int GroupLen, SigLen;
	int Sig, ActSig;
	char line[30];

	GroupLen = 32; SigLen = 16;
	StartGroup = 0x40040510;
	StopGroup = StartGroup+0x1;
	StartVal = 0xd000; StopVal = 0xd100;
	i = ReadData("Tape");
	// sound(4000);
	// delay(500);
	nosound();
	SigNo = 1;
	ActSig = 3;

	for (c=StartGroup; c<StopGroup; c++) {
		for (i=StartVal; i<StopVal; i++) {

			SigNumToString(c, GroupLen, i, SigLen, SD[SigNo].data);

			// Play signal
			CalcSignal(0x0002, SigNo, 0);
			err = PlayIR(Signal, SIZE);

			if ((i%0x100) == 0) {
				sprintf(line, "Done %lx - %x\0", c, i);
				PrintMsg(line);
			}
		}
	}
	PrintMsg("Finished");
	sound(4000);
	delay(500);
	nosound();

	/* for (i=0; i<150; i++)
		if (SD[ActSig].data[i] != SD[SigNo].data[i])
			printf("Diff at %d\n", i);
	printf("%s\n%s\n", SD[ActSig].data, SD[SigNo].data); */
	return 0;
}

// Converts a signal number into PC IR signal string format
void SigNumToString(long int Group, int Glen, int Sig, int Slen, char *S)
{
	int i, ptr=0;
	long int Temp;

	strcpy(S, "111111110000");
	ptr+=strlen(S);

	for (i=Glen-1; i>=0; i--) {
		Temp = Group >> i;
		if ((Temp&1)==0) { strcpy(&S[ptr], ZERO); ptr+=2;}
		else { strcpy(&S[ptr], ONE); ptr+=4; }
	}

	for (i=Slen-1; i>=0; i--) {
		Temp = Sig >> i;
		if ((Temp&1)==0) { strcpy(&S[ptr], ZERO); ptr+=2; }
		else { strcpy(&S[ptr], ONE); ptr+=4; }
	}

	strcpy(&S[ptr], ONE); ptr+=4;

	if (ptr!=150) {
		for (i=ptr; i<150; i++) S[i] = '0';
	}
	S[149]='\0';

	return;
}
